#ifndef __CFileStream__
#define __CFileStream__

//	===========================================================================

#include "../Basics/CSystemString.hpp"
using Exponent::Basics::CSystemString;

//	===========================================================================

namespace Exponent
{
	namespace IO
	{
		/**
		 * @class CFileStream CFileStream.hpp
		 * @brief Binary file stream, writes platform independant (ie endian swapped) data in binary format
		 *
		 * Wrapper class around FILE read and write
		 *
		 * @date 11/04/2006
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @version 1.0.1 Added advance stream function
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CFileStream.hpp,v 1.9 2007/02/08 21:06:44 paul Exp $
		 */
		class CFileStream : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			/**
			 * @enum EStreamMode
			 * @brief Enumeration of stream mmodes
			 */
			enum EStreamMode
			{
				e_input = 0,		/**< Input mode */
				e_output,			/**< Output mode */
			};

//	===========================================================================

			/**
			 * Construction
			 */
			CFileStream();

			/**
			 * Construction which opens a binary file
			 * @param filename The name of the file
			 * @param status The status with which to open the file
			 * @param fileIsLittleEndian If true bytes are little endian order, else big endian
			 */
			CFileStream(const CSystemString &filename, const EStreamMode status, const bool fileIsLittleEndian = true);

			/**
			 * Destruction
			 */
			virtual ~CFileStream();

//	===========================================================================

			/**
			 * Open the stream
			 * @param path The path to the file to read / write
			 * @param mode The streaming mode to use
			 * @param fileIsLittleEndian If true bytes are little endian order, else big endian
			 * @retval bool True if opened properly, false otherwise
			 */
			bool openStream(const CSystemString &path, const EStreamMode mode, const bool fileIsLittleEndian = true);

			/**
			 * Close the stream
			 */
			void closeStream();

//	===========================================================================

			/**
			 * Is the stream open
			 * @retval bool True if stream open, false otherwise
			 */
			bool isStreamOpen() const { return m_streamIsOpen; }

			/**
			 * Valid for output
			 * @retval bool True if file is open for output, false otherwise
			 */
			bool validForOutput();

			/**
			 * Valid for output
			 * @retval bool True if file is open for input, false otherwise
			 */
			bool validForInput();

			/**
			 * Set the stream endianess
			 * @param fileIsLittleEndian True if little endian, false otherwise
			 */
			void setStreamEndianess(const bool fileIsLittleEndian = true);

//	===========================================================================

			/**
			 * Move to a stream postion
			 * @param position The position to mvoe to
			 * @retval bool True if moved, false otherwise
			 */
			bool moveToStreamPosition(const long position);

			/**
			 * Get the stream position
			 * @retval long The current file position, -1 on error
			 * @see ftell
			 */
			long getStreamPosition(); 

			/**
			 * Advance the stream, can be used to move backwards by using a negative value for amount
			 * @param amount The amount to move on by
			 * @retval bool True if moved, false otherwise
			 */
			bool advanceStream(const long amount);

			/**
			 * Move to stream beggining
			 */
			bool moveToStreamStart();

			/**
			 * Get the stream size
			 * @retval long -1 on error or if stream open for write, length in bytes otherwise
			 */
			long getStreamSize();

//	===========================================================================

			/**
			 * Write a block of data to the stream
			 * @param data The data to write
			 * @param numberOfBytes The size of the data in bytes
			 * @retval bool True if written to the stream, false otherwise
			 */
			bool writeDataToStream(const void *data, const size_t numberOfBytes);

			/**
			 * Read a block of data from the stream
			 * @param data The data to read in to on return holds the data. Must be at least numberOfBytes in size
			 * @param numberOfBytes The size of the data in bytes
			 * @retval bool True if read from the stream, false otherwise
			 */
			bool readDataFromStream(void *data, const size_t numberOfBytes);

			/**
			 * Read a block of shorts from the stream
			 * @param data The data to read in to, on return holds the data. Must be at least numberOfElements in size
			 * @param numberOfElements The size of the data in elements
			 * @retval bool True if read from the stream, false otherwise
			 */
			bool readShortsFromStream(short *data, const unsigned long numberOfElements);

//	===========================================================================

			/**
			 * Stream output of a character (BYTE)
			 * @param byte The character to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator << (const char &byte);

			/**
			 * Stream output of a character (BYTE)
			 * @param byte The character to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator << (const unsigned char &byte);

			/**
			 * Stream output of a character (BYTE)
			 * @param byte The bool to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator << (const bool &byte);

			/**
			 * Stream output of a character (BYTE)
			 * @param byte The character to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator << (const short &byte);

			/**
			 * Stream output of a character (BYTE)
			 * @param byte The character to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator << (const unsigned short &byte);

			/**
			 * Stream output of a long (32 bit)
			 * @param value The value to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator << (const long &value);

			/**
			 * Stream output of a long (32 bit)
			 * @param value The long to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator << (const unsigned long &value);

			/**
			 * Stream output of a long (32 bit)
			 * @param value The value to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator << (const float &value);

			/**
			 * Stream output of a long (64 bit)
			 * @param value The value to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator << (const double &value);

//	===========================================================================

			/**
			 * Stream output of a character (BYTE)
			 * @param byte The character to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator >> (char &byte);

			/**
			 * Stream output of a character (BYTE)
			 * @param byte The character to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator >> (unsigned char &byte);

			/**
			 * Stream output of a character (BYTE)
			 * @param byte The bool to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator >> (bool &byte);

			/**
			 * Stream output of a character (BYTE)
			 * @param byte The character to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator >> (short &byte);

			/**
			 * Stream output of a character (BYTE)
			 * @param byte The character to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator >> (unsigned short &byte);

			/**
			 * Stream output of a long (32 bit)
			 * @param value The value to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator >> (long &value);

			/**
			 * Stream output of a long (32 bit)
			 * @param value The long to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator >> (unsigned long &value);

			/**
			 * Stream output of a long (32 bit)
			 * @param value The value to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator >> (float &value);

			/**
			 * Stream output of a long (64 bit)
			 * @param value The value to write
			 * @retval CFileStream & A reference to this
			 */
			CFileStream &operator >> (double &value);

//	===========================================================================

			/**
			 * Has an error occurred
			 * @retval bool True if an error has occurred recently
			 */
			bool hasErrorOccurred() const { return m_errorState; }

			/**
			 * Have we reached the end of the file
			 * @retval bool True if reach eof, flse otherwise
			 */
			bool hasReachEndOfFile() const;

//	===========================================================================

		protected:

//	===========================================================================

			/**
			 * Write a 16 bit value swapping byte order as necessary
			 * @param value The value to write
			 */
			void write16Bits(const void *value);

			/**
			 * Write a 32 bit value swapping byte order as necessary
			 * @param value The value to write
			 */
			void write32Bits(const void *value);

			/**
			 * Write a 64 bit value swapping byte order as necessary
			 * @param value The value to write
			 */
			void write64Bits(const void *value);

//	===========================================================================

			/**
			 * Read a 16 bit value swapping byte order as necessary
			 * @param value The value to read
			 */
			void read16Bits(void *value);

			/**
			 * Read a 32 bit value swapping byte order as necessary
			 * @param value The value to read
			 */
			void read32Bits(void *value);

			/**
			 * Read a 64 bit value swapping byte order as necessary
			 * @param value The value to read
			 */
			void read64Bits(void *value);

//	===========================================================================

			/**
			 * Should swap bits
			 * @retval bool True if bits need to be swapped
			 */
			bool swapBits() const;

//	===========================================================================

			const static unsigned int CFILESTREAM_8BIT_SIZE  = 1;		/**< Size of 8 bit */
			const static unsigned int CFILESTREAM_16BIT_SIZE = 2;		/**< Size of 16 bit */
			const static unsigned int CFILESTREAM_32BIT_SIZE = 4;		/**< Size of eight bit */
			const static unsigned int CFILESTREAM_64BIT_SIZE = 8;		/**< Size of eight bit */

//	===========================================================================

			FILE *m_file;												/**< The file we are writing to */
			EStreamMode m_streamStatus;									/**< Status of the stream */
			bool m_streamIsOpen;										/**< Is the file open */
			bool m_errorState;											/**< Has an error occurred */
			bool m_fileIsLittleEndian;									/**< Are bytes little or big endian */
		};
	}
}
#endif	// End of CFileStream.hpp